/**
  Copyright (c) 2010 Freescale Semiconductor
  Freescale Confidential Proprietary
  
  \file  	  DMAHandler.c
  \brief	  This is the DMA Handler Library Driver File
  \brief	  Library for Controlling the DMA blit operations
  \brief	  Combines DCU and DMA drivers functionality
  \author	  Freescale Semiconductor
  \author	  MSG
  \author	  IM, b06623
  \version	  2.0
  \revision	  $Revision: 32 $
  \date  	  $Date: 2010-09-10 14:56:50 -0500 (Fri, 10 Sep 2010) $  
  
  * History:  18/March/2008 - Initial Version
  *   		  12/May/2009 - MISRA Check, Release
  							improved, simplicity etc etc
 			  25/May/2010	- Context Switching support 							

  * MISRA VIOLATIONS:
	- [ MISRA 11.2 ]
	- [ MISRA 16.9 ]

* Copyright (c) 2010, Freescale, Inc.  All rights reserved.
*
*
* No part of this document must be reproduced in any form - including copied,
* transcribed, printed or by any electronic means - without specific written
* permission from Freescale Semiconductor.
*
  
*/

#include	 "DMAHandler.h"


uint16_t			DMAHR_CurrentCH;
uint8_t 			DMAHR_StopScheduler; 
uint8_t 			DMAHR_LSBFVSCH;
#ifdef DCULITE
uint8_t 			DMAHR_LSBFVSCHlite;
#endif 
uint8_t				DMAHR_DF_Flag;
GALLO_CallbackType	DMAHR_DF_Callback;

static void DMAHR_CallbackDF(void);

static void DMAHR_CallbackVSBLANK(void);
static void DMAHR_CallbackLSBFVS(void);

#ifdef DCULITE
static void DMAHR_CallbackVSBLANKLite(void);
static void DMAHR_CallbackLSBFVSLite(void);
#endif


/**
* \brief	DMAHR_Initt - Initializes the graphics library control variables
* \author	IM, b06623
* \param	void
* \return	void
* \todo
*/
void DMAHR_Init(void)
{
	uint8_t dcuCtx;
	dcuCtx = DCU_GetCurrentDCU();
	
   	DMAHR_CurrentCH = DMAHR_START_CH;
   	DMAHR_StopScheduler = 0u;
   	DMAHR_LSBFVSCH = DMAHR_NOCHNL;
	
	DCU_SelectDCU(DCU_DCU);
	/* VIOLATION TO [ MISRA 16.9 ] Rationale: This is the only way to configure a callback */  
   	DCU_SetCallbackVSBLANK(DMAHR_CallbackVSBLANK);
	/* VIOLATION TO [ MISRA 16.9 ] Rationale: This is the only way to configure a callback */  
   	DCU_SetCallbackLSBFVS(DMAHR_CallbackLSBFVS);   	

#ifdef DCULITE	
	DMAHR_LSBFVSCHlite = DMAHR_NOCHNL;
	DCU_SelectDCU(DCU_DCULITE);
	/* VIOLATION TO [ MISRA 16.9 ] Rationale: This is the only way to configure a callback */  
   	DCU_SetCallbackVSBLANK(DMAHR_CallbackVSBLANKLite);
   	DCU_SetCallbackLSBFVS(DMAHR_CallbackLSBFVSLite);  
#endif   	   
   	
   	DMAHR_DF_Flag = 0u;
   	/* VIOLATION to [MISRA 11.2] Rationale: NULL pointers used to identify empty objects */
   	/* VIOLATION TO [ MISRA 16.9 ] Rationale: This is the only way to configure an (empty) callback */ 
   	DMAHR_DF_Callback = NULL_PTR;
   	DCU_SelectDCU(dcuCtx);
}

/**
* \brief	DMAHR_Suspend - Suspends any tasking of the DMA channels
* \author	IM, b06623
* \param	void
* \return	void
* \todo
*/
void DMAHR_Suspend(void)
{
    DMAHR_StopScheduler = 1u;
}

/**
* \brief	DMAHR_ScheduleDefrag - Schedules a defragmentation whem the DMA
* \brief	channels are all idle
* \author	IM, b06623
* \param	GALLO_CallbackType cb, is the function called when the defragmentation is done
* \return	void
* \todo
*/
void DMAHR_ScheduleDefrag(DMAHR_Callback_t cb)
{
    DMAHR_DF_Flag = 1u;
    /* VIOLATION TO [ MISRA 16.9 ] Rationale: This is the only way to configure a callback */ 
    DMAHR_DF_Callback = cb;
}

/**
* \brief	DMAHR_Resume - Resumes the sch. tasking of the DMA channels
* \author	IM, b06623
* \param	void
* \return	void
* \todo
*/
void DMAHR_Resume(void)
{
    DMAHR_StopScheduler = 0u;
}

/**
* \brief	DMAHR_GetNextFreeChannel - Obtains the next free in the arbitration logic.
* \author	IM, b06623
* \param	void
* \return	void
* \todo
*/
uint8_t DMAHR_GetNextFreeChannel(void)
{
    uint8_t 	i;

	for(i = DMAHR_START_CH; i<= DMAHR_END_CH; i++)
	{
		if(DMA_getChannelStatus(i) == DMA_CH_IDLE)
		{
			return i;
		}
	}
	return DMAHR_NOCHNL;
}

/**
* \brief	DMAHR_TriggerChannels: This internal functions is in charge of triggering all the ready
* \brief	channels
* \author	IM, b06623
* \param	void
* \return	void
* \todo
*/
static void DMAHR_TriggerChannels(uint8_t readyCtx)
{
    uint8_t i;
    uint8_t idle;

    idle = 1;
    
    /* Check if the DMA Handler is on hold, or a DMA task is still running */
    if(DMAHR_StopScheduler == 1)
    {
    	return;
    }
    
    
	for(i = DMAHR_START_CH; i<= DMAHR_END_CH; i++)
	{
		if(DMA_getChannelStatus(i) == readyCtx)
		{
			idle = 0;
			DMA_Start(i);
		}
	}    
	
	/* Defrag is pending */
	if(DMAHR_DF_Flag == 1u)
	{
		/* no tasks were performed, free way to defrag */
		if(idle)
		{
			/* It can be performed at VSBLANK, only. Verify */
#ifdef DCULITE			
			if(readyCtx == DMA_CH_READY || readyCtx == DMA_CH_READY_LITE)
#else
			if(readyCtx == DMA_CH_READY)
#endif			
			{
	    		DMAHR_Suspend();
        		/* VIOLATION TO [ MISRA 16.9 ] Rationale: This is the only way to configure a callback */  
	    		GALLO_DeFragment(DMAHR_START_CH, DMAHR_CallbackDF);
	    		DMAHR_DF_Flag = 0;
	    	}	
	    }    
	}	    
}


static void DMAHR_CallbackVSBLANK(void)
{
	DMAHR_TriggerChannels(DMA_CH_READY);
}

static void DMAHR_CallbackLSBFVS(void)
{
	DMAHR_TriggerChannels(DMA_CH_READY_LSBFVS);
}

#ifdef DCULITE
static void DMAHR_CallbackVSBLANKLite(void)
{
	DMAHR_TriggerChannels(DMA_CH_READY_LITE);
}

static void DMAHR_CallbackLSBFVSLite(void)
{
	DMAHR_TriggerChannels(DMA_CH_READY_LSBFVS_LITE);
}
#endif

/* task called after defrag, to de init the channel and set everything OK. */
static void DMAHR_CallbackDF(void)
{
     /* VIOLATION TO [ MISRA 16.9 ] Rationale: This is the only way to test an empty callback */ 
    if(DMAHR_DF_Callback != NULL_PTR)
    {
		DMAHR_DF_Callback();
    }
    DMAHR_Resume();
}
